/*
 * Decompiled with CFR 0.152.
 */
package mekanism.common.lib.radiation;

import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.function.IntSupplier;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import mekanism.api.Coord4D;
import mekanism.api.text.EnumColor;
import mekanism.common.Mekanism;
import mekanism.common.capabilities.Capabilities;
import mekanism.common.config.MekanismConfig;
import mekanism.common.lib.HashList;
import mekanism.common.lib.math.voxel.Chunk3D;
import mekanism.common.lib.radiation.Meltdown;
import mekanism.common.lib.radiation.RadiationSource;
import mekanism.common.lib.radiation.capability.IRadiationEntity;
import mekanism.common.lib.radiation.capability.IRadiationShielding;
import mekanism.common.network.PacketRadiationData;
import mekanism.common.registries.MekanismParticleTypes;
import mekanism.common.registries.MekanismSounds;
import mekanism.common.util.CapabilityUtils;
import mekanism.common.util.EnumUtils;
import mekanism.common.util.MekanismUtils;
import net.minecraft.entity.Entity;
import net.minecraft.entity.LivingEntity;
import net.minecraft.entity.player.PlayerEntity;
import net.minecraft.entity.player.ServerPlayerEntity;
import net.minecraft.inventory.EquipmentSlotType;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.CompoundNBT;
import net.minecraft.nbt.INBT;
import net.minecraft.nbt.ListNBT;
import net.minecraft.particles.IParticleData;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.SoundEvent;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import net.minecraft.world.storage.DimensionSavedDataManager;
import net.minecraft.world.storage.WorldSavedData;
import net.minecraftforge.common.capabilities.ICapabilityProvider;
import net.minecraftforge.common.util.LazyOptional;
import net.minecraftforge.event.entity.living.LivingEvent;
import net.minecraftforge.eventbus.api.SubscribeEvent;
import net.minecraftforge.fml.server.ServerLifecycleHooks;

public class RadiationManager {
    private static final String DATA_HANDLER_NAME = "radiation_manager";
    private static final IntSupplier MAX_RANGE = () -> MekanismConfig.general.radiationChunkCheckRadius.get() * 16;
    private static final Random RAND = new Random();
    public static final double BASELINE = 1.0E-7;
    public static final double MIN_MAGNITUDE = 1.0E-5;
    private boolean loaded;
    private final Map<Chunk3D, Map<Coord4D, RadiationSource>> radiationMap = new Object2ObjectOpenHashMap();
    private final Map<ResourceLocation, List<Meltdown>> meltdowns = new Object2ObjectOpenHashMap();
    private final Map<UUID, RadiationScale> playerExposureMap = new Object2ObjectOpenHashMap();
    private RadiationScale clientRadiationScale = RadiationScale.NONE;
    @Nullable
    private RadiationDataHandler dataHandler;

    public double getRadiationLevel(Entity entity) {
        return this.getRadiationLevel(new Coord4D(entity));
    }

    public double getRadiationLevel(Coord4D coord) {
        Set<Chunk3D> checkChunks = new Chunk3D(coord).expand(MekanismConfig.general.radiationChunkCheckRadius.get());
        double level = 1.0E-7;
        for (Chunk3D chunk : checkChunks) {
            if (!this.radiationMap.containsKey((Object)chunk)) continue;
            for (RadiationSource src : this.radiationMap.get((Object)chunk).values()) {
                if (!(src.getPos().distanceTo(coord) <= (double)MAX_RANGE.getAsInt())) continue;
                double add = this.computeExposure(coord, src);
                level += add;
            }
        }
        return level;
    }

    public void radiate(Coord4D coord, double magnitude) {
        RadiationSource src;
        if (!MekanismConfig.general.radiationEnabled.get()) {
            return;
        }
        Chunk3D chunk = new Chunk3D(coord);
        boolean found = false;
        if (this.radiationMap.containsKey((Object)chunk) && (src = this.radiationMap.get((Object)chunk).get(coord)) != null) {
            src.radiate(magnitude);
            found = true;
        }
        if (!found) {
            this.radiationMap.computeIfAbsent(new Chunk3D(coord), c -> new Object2ObjectOpenHashMap()).put(coord, new RadiationSource(coord, magnitude));
        }
    }

    public void radiate(LivingEntity entity, double magnitude) {
        if (!MekanismConfig.general.radiationEnabled.get()) {
            return;
        }
        if (!(entity instanceof PlayerEntity) || MekanismUtils.isPlayingMode((PlayerEntity)entity)) {
            entity.getCapability(Capabilities.RADIATION_ENTITY_CAPABILITY).ifPresent(c -> c.radiate(magnitude * (1.0 - Math.min(1.0, this.getRadiationResistance(entity)))));
        }
    }

    public void createMeltdown(World world, BlockPos minPos, BlockPos maxPos, double magnitude, double chance) {
        this.meltdowns.computeIfAbsent(world.func_234923_W_().func_240901_a_(), id -> new ArrayList()).add(new Meltdown(world, minPos, maxPos, magnitude, chance));
    }

    public void clearSources() {
        this.radiationMap.clear();
    }

    private double computeExposure(Coord4D coord, RadiationSource source) {
        return source.getMagnitude() / Math.max(1.0, Math.pow(coord.distanceTo(source.getPos()), 2.0));
    }

    private double getRadiationResistance(LivingEntity entity) {
        double resistance = 0.0;
        for (EquipmentSlotType type : EnumUtils.ARMOR_SLOTS) {
            ItemStack stack = entity.func_184582_a(type);
            Optional shielding = CapabilityUtils.getCapability((ICapabilityProvider)stack, Capabilities.RADIATION_SHIELDING_CAPABILITY, null).resolve();
            if (!shielding.isPresent()) continue;
            resistance += ((IRadiationShielding)shielding.get()).getRadiationShielding();
        }
        return resistance;
    }

    public void setClientScale(RadiationScale scale) {
        this.clientRadiationScale = scale;
    }

    public RadiationScale getClientScale() {
        return this.clientRadiationScale;
    }

    public void tickClient(PlayerEntity player) {
        if (this.clientRadiationScale != RadiationScale.NONE && player.field_70170_p.func_201674_k().nextInt(2) == 0) {
            int count = player.field_70170_p.func_201674_k().nextInt(this.clientRadiationScale.ordinal() * MekanismConfig.client.radiationParticleCount.get());
            int radius = MekanismConfig.client.radiationParticleRadius.get();
            for (int i = 0; i < count; ++i) {
                double x = player.func_226277_ct_() + player.field_70170_p.func_201674_k().nextDouble() * (double)radius * 2.0 - (double)radius;
                double y = player.func_226278_cu_() + player.field_70170_p.func_201674_k().nextDouble() * (double)radius * 2.0 - (double)radius;
                double z = player.func_226281_cx_() + player.field_70170_p.func_201674_k().nextDouble() * (double)radius * 2.0 - (double)radius;
                player.field_70170_p.func_195594_a((IParticleData)MekanismParticleTypes.RADIATION.getParticleType(), x, y, z, 0.0, 0.0, 0.0);
            }
        }
    }

    public void tickServer(ServerPlayerEntity player) {
        this.updateEntityRadiation((LivingEntity)player);
    }

    public void updateEntityRadiation(LivingEntity entity) {
        if (!MekanismConfig.general.radiationEnabled.get()) {
            return;
        }
        LazyOptional radiationCap = entity.getCapability(Capabilities.RADIATION_ENTITY_CAPABILITY);
        if (entity.field_70170_p.func_201674_k().nextInt(20) == 0) {
            double magnitude = this.getRadiationLevel(new Coord4D((Entity)entity));
            if (magnitude > 1.0E-7 && (!(entity instanceof PlayerEntity) || MekanismUtils.isPlayingMode((PlayerEntity)entity))) {
                this.radiate(entity, magnitude / 3600.0);
            }
            radiationCap.ifPresent(IRadiationEntity::decay);
            if (entity instanceof ServerPlayerEntity) {
                ServerPlayerEntity player = (ServerPlayerEntity)entity;
                RadiationScale scale = RadiationScale.get(magnitude);
                if (this.playerExposureMap.get(player.func_110124_au()) != scale) {
                    this.playerExposureMap.put(player.func_110124_au(), scale);
                    Mekanism.packetHandler.sendTo(PacketRadiationData.create(scale), player);
                }
            }
        }
        radiationCap.ifPresent(c -> c.update(entity));
    }

    public void tickServerWorld(World world) {
        ResourceLocation dimension;
        if (!MekanismConfig.general.radiationEnabled.get()) {
            return;
        }
        if (!this.loaded) {
            this.createOrLoad();
        }
        if (this.meltdowns.containsKey(dimension = world.func_234923_W_().func_240901_a_())) {
            this.meltdowns.get(dimension).removeIf(Meltdown::update);
        }
    }

    public void tickServer() {
        if (!MekanismConfig.general.radiationEnabled.get()) {
            return;
        }
        if (RAND.nextInt(20) == 0) {
            for (Map<Coord4D, RadiationSource> set : this.radiationMap.values()) {
                Iterator<Map.Entry<Coord4D, RadiationSource>> iter = set.entrySet().iterator();
                while (iter.hasNext()) {
                    Map.Entry<Coord4D, RadiationSource> entry = iter.next();
                    if (entry.getValue().decay()) {
                        iter.remove();
                    }
                    this.dataHandler.func_76185_a();
                }
            }
        }
    }

    public void createOrLoad() {
        if (this.dataHandler == null) {
            DimensionSavedDataManager savedData = ServerLifecycleHooks.getCurrentServer().func_241755_D_().func_217481_x();
            this.dataHandler = (RadiationDataHandler)savedData.func_215752_a(RadiationDataHandler::new, DATA_HANDLER_NAME);
            this.dataHandler.setManager(this);
            this.dataHandler.syncManager();
        }
        this.loaded = true;
    }

    public void reset() {
        this.clearSources();
        this.playerExposureMap.clear();
        this.meltdowns.clear();
        this.dataHandler = null;
        this.loaded = false;
    }

    public void resetClient() {
        this.clientRadiationScale = RadiationScale.NONE;
    }

    public void resetPlayer(UUID uuid) {
        this.playerExposureMap.remove(uuid);
    }

    @SubscribeEvent
    public void onLivingUpdate(LivingEvent.LivingUpdateEvent event) {
        World world = event.getEntityLiving().func_130014_f_();
        if (!world.func_201670_d() && !(event.getEntityLiving() instanceof PlayerEntity) && world.func_201674_k().nextInt() % 20 == 0) {
            this.updateEntityRadiation(event.getEntityLiving());
        }
    }

    public static class RadiationDataHandler
    extends WorldSavedData {
        public RadiationManager manager;
        public List<RadiationSource> loadedSources;

        public RadiationDataHandler() {
            super(RadiationManager.DATA_HANDLER_NAME);
        }

        public void setManager(RadiationManager m) {
            this.manager = m;
        }

        public void syncManager() {
            if (this.loadedSources != null && MekanismConfig.general.radiationEnabled.get()) {
                for (RadiationSource source : this.loadedSources) {
                    Chunk3D chunk = new Chunk3D(source.getPos());
                    this.manager.radiationMap.computeIfAbsent(chunk, c -> new Object2ObjectOpenHashMap()).put(source.getPos(), source);
                }
            }
        }

        public void func_76184_a(@Nonnull CompoundNBT nbtTags) {
            if (nbtTags.func_150297_b("radList", 9)) {
                ListNBT list = nbtTags.func_150295_c("radList", 10);
                this.loadedSources = new HashList<RadiationSource>();
                for (int i = 0; i < list.size(); ++i) {
                    this.loadedSources.add(RadiationSource.load(list.func_150305_b(0)));
                }
            }
        }

        @Nonnull
        public CompoundNBT func_189551_b(@Nonnull CompoundNBT nbtTags) {
            ListNBT list = new ListNBT();
            for (Map map : this.manager.radiationMap.values()) {
                for (RadiationSource source : map.values()) {
                    CompoundNBT compound = new CompoundNBT();
                    source.write(compound);
                    list.add((Object)compound);
                }
            }
            nbtTags.func_218657_a("radList", (INBT)list);
            return nbtTags;
        }
    }

    public static enum RadiationScale {
        NONE,
        LOW,
        MEDIUM,
        ELEVATED,
        HIGH,
        EXTREME;

        private static final double LOG_BASELINE;
        private static final double LOG_MAX;
        private static final double SCALE;

        public static RadiationScale get(double magnitude) {
            if (magnitude < 1.0E-5) {
                return NONE;
            }
            if (magnitude < 0.001) {
                return LOW;
            }
            if (magnitude < 0.1) {
                return MEDIUM;
            }
            if (magnitude < 10.0) {
                return ELEVATED;
            }
            if (magnitude < 100.0) {
                return HIGH;
            }
            return EXTREME;
        }

        public static EnumColor getSeverityColor(double magnitude) {
            if (magnitude <= 1.0E-7) {
                return EnumColor.BRIGHT_GREEN;
            }
            if (magnitude < 1.0E-5) {
                return EnumColor.GRAY;
            }
            if (magnitude < 0.001) {
                return EnumColor.YELLOW;
            }
            if (magnitude < 0.1) {
                return EnumColor.ORANGE;
            }
            if (magnitude < 10.0) {
                return EnumColor.RED;
            }
            return EnumColor.DARK_RED;
        }

        public static double getScaledDoseSeverity(double magnitude) {
            if (magnitude < 1.0E-5) {
                return 0.0;
            }
            return Math.min(1.0, Math.max(0.0, (-LOG_BASELINE + Math.log10(magnitude)) / SCALE));
        }

        public SoundEvent getSoundEvent() {
            switch (this) {
                case LOW: {
                    return (SoundEvent)MekanismSounds.GEIGER_SLOW.get();
                }
                case MEDIUM: {
                    return (SoundEvent)MekanismSounds.GEIGER_MEDIUM.get();
                }
                case ELEVATED: 
                case HIGH: {
                    return (SoundEvent)MekanismSounds.GEIGER_ELEVATED.get();
                }
                case EXTREME: {
                    return (SoundEvent)MekanismSounds.GEIGER_FAST.get();
                }
            }
            return null;
        }

        static {
            LOG_BASELINE = Math.log10(1.0E-5);
            LOG_MAX = Math.log10(100.0);
            SCALE = LOG_MAX - LOG_BASELINE;
        }
    }
}

